## Lab session: segmentation by dynamic programming (II)

## Initialization (matrix J): naive implementation in O(n^3)
getJnaive <- function(y) {
    n <- length(y)
    J <- matrix(NA, ncol=n, nrow=n)
    for (ii in 1:n) {
        for(jj in ii:n) {
            avg <- mean(y[ii:jj])
            diffs <- y[ii:jj]-avg
            J[ii,jj] <- sum(diffs^2)
        } ## for (jj ...
    } ## for (ii ...
    return(J)
}

## A less naive implementation in O(n^2)
getJ <- function(y) {
    n <- length(y)
    S <- c(0, cumsum(y))
    T <- c(0, cumsum(y^2))
    J <- matrix(NA, ncol=n, nrow=n)
    for (ii in 1:n) {
        for(jj in ii:n) {
            J[ii, jj] <- T[jj+1]-T[ii] - (S[jj+1]-S[ii])^2/(jj-ii+1)
        } ## for (jj ...
    } ## for (ii ...
    return(J)
}

## step 2: induction 
getVandBkp <- function(J, K) {  ## get 'V'  and 'bkp'
    n <- nrow(J)
    V <- matrix(numeric((K+1)*n), ncol=n)
    ## V[i,j] is the best RSE for segmenting intervals 1 to j
    ## with at most i-1 change points
    bkp <-   matrix(integer(K*n), ncol = n)
    ## bkp[i, j] is the *last* bkp of the best segmentation of [1:j] in i segments
    
    ## With no change points (i=1), V[i,j] is just the precomputed RSE
    ## for intervals 1 to j
    V[1, ] <- J[1, ]
    
    ## Then we apply the recursive formula
    KK <- seq(length=K)
    for (ki in KK){
        for (jj in (ki+seq(length=n-ki))){
            idxh <- ki:(jj-1)
            obj <- V[ki, idxh] + J[idxh+1, jj]
            val <- min(obj)
            ind <- which.min(obj)
            V[ki+1, jj] <- val
            bkp[ki, jj] <- ind + ki-1L
        }
    }
    res <- list(V=V, bkp=bkp)
    return(res)
}    



## step 3: backtracking
backtrack <- function(bkp) {
    ## backtracking to get the optimal segmentation
    K <- nrow(bkp)
    n <- ncol(bkp)
    res.bkp <- list()
    KK <- seq(length=K)
    for (ki in KK){
        res.bkp[[ki]] <- integer(ki)
        res.bkp[[ki]][ki] <- bkp[ki, n]
        if (ki!=1) {
            for (ii in (ki-seq(length=ki-1))) {
                res.bkp[[ki]][ii] <- bkp[ii, res.bkp[[ki]][ii+1]]
            }
        }
    }
    return(res.bkp)
}

## Segmentation by DP 
dpseg <- function(y, K){
    n <- length(y)

    ## Compute the k*k matrix J such that J[i,j] for i<=j is the RSE
    ## when intervals i to j are merged
    J <- getJ(y)

    ## Dynamic programming
    dp <- getVandBkp(J, K)
    V <- dp$V 
    ## V[i,j] is the best RSE for segmenting intervals 1 to j
    ## with at most i-1 change points
    bkp <- dp$bkp
    ## bkp[i, j] is the *last* bkp of the best segmentation of [1:j] in i segments
        
    ## Optimal segmentation
    res.bkp <- backtrack(bkp)

    ## RSE as a function of number of change-points
    res.rse <- V[, n]
    ## Optimal number of change points
    
    list(bkpList=res.bkp, ##<< A list of vectors of change point positions for the best model with k change points, for k=1, 2, ... K
         rse=res.rse, ##<< A vector of K+1 residual squared errors
         V=V) ##<< V[i,j] is the best RSE for segmenting intervals 1 to j
}


